@extends('layouts.app')

@section('title', 'Record Savings Collection')
@section('page-title', 'Record Savings Collection')

@section('content')
<!-- Collection Type Information -->
<div class="alert alert-info mb-3">
    <h6 class="alert-heading"><i class="bi bi-info-circle"></i> Savings Collection Rules</h6>
    <p class="mb-2">Only <strong>RD (Recurring Deposit)</strong> accounts allow collections. Here's why:</p>
    <div class="row mt-3">
        <div class="col-md-4">
            <div class="d-flex align-items-start">
                <span class="badge bg-success me-2">✓</span>
                <div>
                    <strong>RD (Recurring Deposit)</strong>
                    <br><small class="text-muted">Regular installments (daily/weekly/monthly)</small>
                    <br><span class="badge bg-success mt-1">Collections Allowed</span>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="d-flex align-items-start">
                <span class="badge bg-danger me-2">✗</span>
                <div>
                    <strong>FD (Fixed Deposit)</strong>
                    <br><small class="text-muted">One-time lump sum deposit</small>
                    <br><span class="badge bg-secondary mt-1">No Collections</span>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="d-flex align-items-start">
                <span class="badge bg-danger me-2">✗</span>
                <div>
                    <strong>MIS (Monthly Income)</strong>
                    <br><small class="text-muted">One-time deposit, only payouts</small>
                    <br><span class="badge bg-secondary mt-1">No Collections</span>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="card">
    <div class="card-header bg-primary text-white">
        <h5 class="mb-0"><i class="bi bi-cash-stack"></i> Record RD Collection</h5>
    </div>
    <div class="card-body">
        @if($savings->isEmpty())
            <div class="alert alert-warning">
                <i class="bi bi-exclamation-triangle"></i> <strong>No RD accounts available for collection.</strong>
                <p class="mb-0 mt-2">
                    Only active RD (Recurring Deposit) accounts with pending installments can have collections.
                    <br>FD and MIS accounts are created with a one-time deposit and don't require collections.
                </p>
                <a href="{{ route('partner.savings.create') }}" class="btn btn-sm btn-primary mt-2">
                    <i class="bi bi-plus-circle"></i> Create New RD Account
                </a>
            </div>
        @else
        <form action="{{ route('partner.savings-collections.store') }}" method="POST">
            @csrf
            <div class="mb-3">
                <label for="saving_id" class="form-label">RD Account *</label>
                <select class="form-select @error('saving_id') is-invalid @enderror" id="saving_id" name="saving_id" required>
                    <option value="">Select RD Account</option>
                    @foreach($savings as $saving)
                        @php
                            $freqLabel = match($saving->duration_type) {
                                'daily' => 'days',
                                'weekly' => 'weeks',
                                'monthly' => 'months',
                                default => 'installments'
                            };
                            $freqSingle = match($saving->duration_type) {
                                'daily' => 'Daily',
                                'weekly' => 'Weekly',
                                'monthly' => 'Monthly',
                                default => 'Regular'
                            };
                            $progress = $saving->duration_value > 0 ? round(($saving->collected_months / $saving->duration_value) * 100) : 0;
                        @endphp
                        <option value="{{ $saving->id }}" 
                                data-monthly="{{ $saving->monthly_deposit }}" 
                                data-collected="{{ $saving->collected_months }}"
                                data-total="{{ $saving->duration_value }}"
                                data-remaining="{{ $saving->remaining_months }}"
                                data-frequency="{{ $saving->duration_type }}"
                                data-scheme="{{ $saving->savingsScheme->name ?? 'N/A' }}"
                                data-rate="{{ $saving->savingsScheme->interest_rate ?? 0 }}"
                                {{ old('saving_id') == $saving->id ? 'selected' : '' }}>
                            [RD-{{ $freqSingle }}] {{ $saving->savings_number }} - {{ $saving->customer->name }} 
                            | ₹{{ number_format($saving->monthly_deposit, 2) }}/{{ rtrim($freqLabel, 's') }} 
                            | {{ $saving->collected_months }}/{{ $saving->duration_value }} ({{ $progress }}%)
                        </option>
                    @endforeach
                </select>
                @error('saving_id')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
                <small class="text-muted">
                    <i class="bi bi-lightbulb"></i> Select an RD account to view its collection schedule
                </small>
            </div>

            <!-- RD Account Details -->
            <div id="account-details" class="alert alert-light border" style="display: none;">
                <h6 class="mb-3"><i class="bi bi-card-list"></i> RD Account Details</h6>
                <div class="row">
                    <div class="col-md-3">
                        <small class="text-muted">Frequency</small>
                        <div><strong id="detail-frequency">-</strong></div>
                    </div>
                    <div class="col-md-3">
                        <small class="text-muted">Installment Amount</small>
                        <div><strong id="detail-installment">-</strong></div>
                    </div>
                    <div class="col-md-3">
                        <small class="text-muted">Progress</small>
                        <div><strong id="detail-progress">-</strong></div>
                    </div>
                    <div class="col-md-3">
                        <small class="text-muted">Scheme</small>
                        <div><strong id="detail-scheme">-</strong></div>
                    </div>
                </div>
            </div>

            <!-- Due Date Selector -->
            <div id="due-date-section" class="mb-3" style="display: none;">
                <label for="due_date_selector" class="form-label">Select Due Date (Optional)</label>
                <select class="form-select" id="due_date_selector">
                    <option value="">-- Select Due Date to Auto-fill Amount --</option>
                </select>
                <small class="text-muted">Select a due date to automatically fill the installment amount, or enter a custom amount below</small>
            </div>

            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="amount" class="form-label">Amount *</label>
                    <input type="number" step="0.01" class="form-control @error('amount') is-invalid @enderror" 
                           id="amount" name="amount" value="{{ old('amount') }}" required>
                    <small class="text-muted">Select a due date above to auto-fill, or enter custom amount</small>
                    @error('amount')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-6 mb-3">
                    <label for="collection_date" class="form-label">Collection Date *</label>
                    <input type="date" class="form-control @error('collection_date') is-invalid @enderror" 
                           id="collection_date" name="collection_date" value="{{ old('collection_date', date('Y-m-d')) }}" required>
                    @error('collection_date')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>
            </div>
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="payment_method" class="form-label">Payment Method *</label>
                    <select class="form-select @error('payment_method') is-invalid @enderror" id="payment_method" name="payment_method" required>
                        <option value="cash" {{ old('payment_method') === 'cash' ? 'selected' : '' }}>Cash</option>
                        <option value="bank_transfer" {{ old('payment_method') === 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                        <option value="cheque" {{ old('payment_method') === 'cheque' ? 'selected' : '' }}>Cheque</option>
                        <option value="other" {{ old('payment_method') === 'other' ? 'selected' : '' }}>Other</option>
                    </select>
                    @error('payment_method')
                        <div class="invalid-feedback">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-6 mb-3">
                    <label for="reference_number" class="form-label">Reference Number</label>
                    <input type="text" class="form-control" id="reference_number" name="reference_number" value="{{ old('reference_number') }}">
                </div>
            </div>
            <div class="mb-3">
                <label for="notes" class="form-label">Notes</label>
                <textarea class="form-control" id="notes" name="notes" rows="3">{{ old('notes') }}</textarea>
            </div>
            <div class="d-flex gap-2">
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-cash-coin"></i> Record Collection
                </button>
                <a href="{{ route('partner.savings-collections.index') }}" class="btn btn-secondary">
                    <i class="bi bi-x-circle"></i> Cancel
                </a>
            </div>
        </form>
        @endif
    </div>
</div>

<!-- Additional Information -->
<div class="card mt-3">
    <div class="card-header">
        <h6 class="mb-0"><i class="bi bi-question-circle"></i> Why Only RD?</h6>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <h6 class="text-success"><i class="bi bi-check-circle"></i> RD (Recurring Deposit)</h6>
                <ul class="small">
                    <li>Customers deposit <strong>regular installments</strong> (daily/weekly/monthly)</li>
                    <li>Each installment is collected separately over time</li>
                    <li>Collections are tracked against a schedule</li>
                    <li>Interest calculated on cumulative deposits</li>
                </ul>
                <p class="mb-0 small text-muted"><strong>Example:</strong> ₹5,000/month for 12 months = 12 collections</p>
            </div>
            <div class="col-md-6">
                <h6 class="text-secondary"><i class="bi bi-x-circle"></i> FD & MIS</h6>
                <ul class="small">
                    <li><strong>FD:</strong> One-time lump sum deposit at account opening</li>
                    <li><strong>MIS:</strong> One-time deposit, customer receives monthly interest payouts</li>
                    <li>Both are created with full principal amount</li>
                    <li>No recurring collections needed</li>
                </ul>
                <p class="mb-0 small text-muted"><strong>Example:</strong> ₹1,00,000 FD = 1 deposit at opening, no further collections</p>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    console.log('🔄 Savings collection form script loaded');
    
    const savingSelect = document.getElementById('saving_id');
    const dueDateSection = document.getElementById('due-date-section');
    const dueDateSelector = document.getElementById('due_date_selector');
    const amountInput = document.getElementById('amount');
    const collectionDateInput = document.getElementById('collection_date');
    const accountDetails = document.getElementById('account-details');
    let currentSchedule = [];
    
    console.log('Elements found:', {
        savingSelect: !!savingSelect,
        dueDateSection: !!dueDateSection,
        dueDateSelector: !!dueDateSelector,
        amountInput: !!amountInput,
        collectionDateInput: !!collectionDateInput
    });

    // Load schedule when saving is selected
    savingSelect.addEventListener('change', function() {
        const savingId = this.value;
        console.log('📋 Saving changed to:', savingId);
        
        if (savingId) {
            // Show account details
            const selectedOption = this.options[this.selectedIndex];
            const frequency = selectedOption.dataset.frequency || 'monthly';
            const installment = parseFloat(selectedOption.dataset.monthly || 0);
            const collected = parseInt(selectedOption.dataset.collected || 0);
            const total = parseInt(selectedOption.dataset.total || 0);
            const scheme = selectedOption.dataset.scheme || 'N/A';
            const rate = selectedOption.dataset.rate || 0;
            
            const freqLabel = frequency === 'daily' ? 'Daily' : 
                            frequency === 'weekly' ? 'Weekly' : 'Monthly';
            const progress = total > 0 ? Math.round((collected / total) * 100) : 0;
            
            document.getElementById('detail-frequency').textContent = freqLabel;
            document.getElementById('detail-installment').textContent = '₹' + installment.toLocaleString('en-IN', {minimumFractionDigits: 2});
            document.getElementById('detail-progress').textContent = `${collected}/${total} (${progress}%)`;
            document.getElementById('detail-scheme').textContent = `${scheme} (${rate}%)`;
            
            accountDetails.style.display = 'block';
            
            loadSchedule(savingId);
        } else {
            accountDetails.style.display = 'none';
            dueDateSection.style.display = 'none';
            dueDateSelector.innerHTML = '<option value="">-- Select Due Date to Auto-fill Amount --</option>';
        }
        
        // Clear amount and reset date when saving changes
        amountInput.value = '';
        collectionDateInput.value = '{{ date('Y-m-d') }}';
    });
    
    // Handle due date selection
    dueDateSelector.addEventListener('change', function() {
        const selectedIndex = this.value;
        console.log('📅 Due date selected, index:', selectedIndex);
        
        if (selectedIndex !== '' && currentSchedule[selectedIndex]) {
            const installment = currentSchedule[selectedIndex];
            console.log('Selected installment:', installment);
            
            // Auto-fill amount
            amountInput.value = parseFloat(installment.due_amount).toFixed(2);
            
            // Auto-fill collection date with due date
            collectionDateInput.value = installment.due_date;
            
            console.log('✅ Auto-filled - Amount:', amountInput.value, 'Date:', collectionDateInput.value);
        } else {
            // Clear fields if "select" option is chosen
            amountInput.value = '';
            collectionDateInput.value = '{{ date('Y-m-d') }}';
        }
    });
    
    // Check for query parameter
    const urlParams = new URLSearchParams(window.location.search);
    const savingIdParam = urlParams.get('saving_id');
    if (savingIdParam) {
        savingSelect.value = savingIdParam;
        console.log('✅ Pre-selected saving from URL:', savingIdParam);
    }
    
    // Trigger initial load if saving is already selected (from old input or query params)
    if (savingSelect.value) {
        console.log('✅ Pre-selected saving detected:', savingSelect.value);
        loadSchedule(savingSelect.value);
    } else {
        console.log('⚠️ No saving pre-selected');
    }

    // Load schedule from API
    function loadSchedule(savingId) {
        console.log('📅 Loading collection schedule for saving:', savingId);
        dueDateSection.style.display = 'block';
        dueDateSelector.innerHTML = '<option value="">Loading...</option>';

        // Build URL using placeholder and replace it
        const urlTemplate = '{{ route("partner.savings-collections.saving.schedule", 999999) }}';
        const url = urlTemplate.replace('999999', savingId);
        console.log('🔗 Fetching from:', url);
        
        fetch(url)
            .then(response => {
                console.log('📡 Response received:', response.status, response.statusText);
                if (!response.ok) throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                return response.json();
            })
            .then(data => {
                console.log('✅ Schedule loaded successfully:', data.schedule.length, 'installments');
                currentSchedule = data.schedule;
                populateDropdown(data.schedule);
            })
            .catch(error => {
                console.error('❌ Error loading schedule:', error);
                dueDateSelector.innerHTML = '<option value="">Error loading schedule</option>';
            });
    }

    // Populate dropdown with due dates
    function populateDropdown(schedule) {
        console.log('🎨 Populating dropdown with', schedule.length, 'installments');
        
        let options = '<option value="">-- Select Due Date to Auto-fill Amount --</option>';
        
        schedule.forEach((item, index) => {
            // Only show pending and overdue installments
            if (item.status !== 'collected' && parseFloat(item.due_amount) > 0) {
                const dueDate = new Date(item.due_date);
                const formattedDate = dueDate.toLocaleDateString('en-GB', { 
                    day: '2-digit', 
                    month: 'short', 
                    year: 'numeric' 
                });
                const amount = parseFloat(item.due_amount).toLocaleString('en-IN', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                });
                
                const statusBadge = item.status === 'overdue' ? '⚠️ Overdue' : 'Pending';
                
                options += `<option value="${index}">
                    ${formattedDate} - ₹${amount} (${statusBadge})
                </option>`;
            }
        });
        
        dueDateSelector.innerHTML = options;
        console.log('✅ Dropdown populated');
    }
    
    console.log('✓ Savings collection form with dropdown due date selector initialized');
});
</script>
@endpush
@endsection

