<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Partner extends Model
{
    protected $fillable = [
        'name',
        'email',
        'phone',
        'whatsapp',
        'address',
        'company_name',
        'logo',
        'status',
        // Customer Number Format Settings
        'customer_prefix',
        'customer_suffix',
        'customer_number_type',
        'customer_last_number',
        // Loan Number Format Settings
        'loan_prefix',
        'loan_suffix',
        'loan_number_type',
        'loan_last_number',
        // Savings Number Format Settings
        'savings_prefix',
        'savings_suffix',
        'savings_number_type',
        'savings_last_number',
    ];

    public function users(): HasMany
    {
        return $this->hasMany(User::class);
    }

    public function customers(): HasMany
    {
        return $this->hasMany(Customer::class);
    }

    public function schemes(): HasMany
    {
        return $this->hasMany(Scheme::class);
    }

    public function loanSchemes(): HasMany
    {
        return $this->hasMany(LoanScheme::class);
    }

    public function savingsSchemes(): HasMany
    {
        return $this->hasMany(SavingsScheme::class);
    }

    public function loans(): HasMany
    {
        return $this->hasMany(Loan::class);
    }

    public function savings(): HasMany
    {
        return $this->hasMany(Saving::class);
    }

    public function repayments(): HasMany
    {
        return $this->hasMany(Repayment::class);
    }

    public function penalties(): HasMany
    {
        return $this->hasMany(Penalty::class);
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscription()
    {
        return $this->subscriptions()
            ->where('status', 'active')
            ->where('end_date', '>=', now())
            ->latest()
            ->first();
    }

    public function hasActiveSubscription(): bool
    {
        return $this->activeSubscription() !== null;
    }

    /**
     * Generate customer code based on partner settings
     */
    public function generateCustomerCode(): string
    {
        $prefix = $this->customer_prefix ?? '';
        $suffix = $this->customer_suffix ?? '';
        
        if ($this->customer_number_type === 'random') {
            $number = str_pad(rand(10000, 99999), 5, '0', STR_PAD_LEFT);
        } else {
            $this->customer_last_number = ($this->customer_last_number ?? 0) + 1;
            $this->save();
            $number = str_pad($this->customer_last_number, 5, '0', STR_PAD_LEFT);
        }
        
        return $prefix . $number . $suffix;
    }

    /**
     * Generate loan number based on partner settings
     */
    public function generateLoanNumber(): string
    {
        $prefix = $this->loan_prefix ?? '';
        $suffix = $this->loan_suffix ?? '';
        
        if ($this->loan_number_type === 'random') {
            $number = str_pad(rand(10000, 99999), 5, '0', STR_PAD_LEFT);
        } else {
            $this->loan_last_number = ($this->loan_last_number ?? 0) + 1;
            $this->save();
            $number = str_pad($this->loan_last_number, 5, '0', STR_PAD_LEFT);
        }
        
        return $prefix . $number . $suffix;
    }

    /**
     * Generate savings number based on partner settings
     */
    public function generateSavingsNumber(): string
    {
        $prefix = $this->savings_prefix ?? '';
        $suffix = $this->savings_suffix ?? '';
        
        if ($this->savings_number_type === 'random') {
            $number = str_pad(rand(10000, 99999), 5, '0', STR_PAD_LEFT);
        } else {
            $this->savings_last_number = ($this->savings_last_number ?? 0) + 1;
            $this->save();
            $number = str_pad($this->savings_last_number, 5, '0', STR_PAD_LEFT);
        }
        
        return $prefix . $number . $suffix;
    }

    public function generateCollectionNumber(): string
    {
        $prefix = 'COL-';
        $number = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        return $prefix . $number;
    }
}
