<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\Setting;
use Illuminate\Http\Request;
use Razorpay\Api\Api;
use Carbon\Carbon;

class SubscriptionController extends Controller
{
    private $razorpay;

    public function __construct()
    {
        $razorpayKey = Setting::get('razorpay_key') ?: config('services.razorpay.key');
        $razorpaySecret = Setting::get('razorpay_secret') ?: config('services.razorpay.secret');
        
        if (empty($razorpayKey) || empty($razorpaySecret)) {
            // Don't initialize if credentials are missing
            $this->razorpay = null;
        } else {
            $this->razorpay = new Api($razorpayKey, $razorpaySecret);
        }
    }

    public function index()
    {
        $partner = auth()->user()->partner;
        $activeSubscription = $partner->activeSubscription();
        $subscriptions = Subscription::where('partner_id', $partner->id)
            ->orderBy('created_at', 'desc')
            ->get();

        // Check if Razorpay is configured
        $razorpayConfigured = !empty(Setting::get('razorpay_key')) && !empty(Setting::get('razorpay_secret'));

        return view('partner.subscription.index', compact('partner', 'activeSubscription', 'subscriptions', 'razorpayConfigured'));
    }

    public function create(Request $request)
    {
        $planType = $request->get('plan', 'monthly');
        
        // Calculate amount
        $monthlyPrice = 300;
        if ($planType === 'yearly') {
            $amount = $monthlyPrice * 12 * 0.9; // 10% discount
        } else {
            $amount = $monthlyPrice;
        }

        $partner = auth()->user()->partner;

        // Check if Razorpay is configured
        if (!$this->razorpay) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Payment gateway is not configured. Please contact administrator.');
        }

        // Create Razorpay Order
        $orderData = [
            'receipt' => 'SUB_' . $partner->id . '_' . time(),
            'amount' => (int)($amount * 100), // Amount in paise
            'currency' => 'INR',
            'notes' => [
                'partner_id' => (string)$partner->id,
                'plan_type' => $planType,
                'amount' => (string)$amount,
            ],
        ];

        try {
            $razorpayOrder = $this->razorpay->order->create($orderData);

            // Create subscription record
            $subscription = Subscription::create([
                'partner_id' => $partner->id,
                'plan_type' => $planType,
                'amount' => $amount,
                'razorpay_order_id' => $razorpayOrder['id'],
                'status' => 'pending',
            ]);

            return view('partner.subscription.payment', [
                'subscription' => $subscription,
                'razorpayOrder' => $razorpayOrder,
                'amount' => $amount,
            ]);
        } catch (\Razorpay\Api\Errors\BadRequestError $e) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Payment gateway error: ' . $e->getMessage() . '. Please check Razorpay API credentials.');
        } catch (\Razorpay\Api\Errors\ServerError $e) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Payment gateway server error. Please try again later.');
        } catch (\Exception $e) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Failed to create payment order: ' . $e->getMessage() . '. Please contact administrator if the issue persists.');
        }
    }

    public function callback(Request $request)
    {
        $input = $request->all();

        $subscription = Subscription::where('razorpay_order_id', $input['razorpay_order_id'])->first();

        if (!$subscription) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Subscription not found');
        }

        // Check if Razorpay is configured
        if (!$this->razorpay) {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Payment gateway is not configured. Please contact administrator.');
        }

        try {
            $attributes = [
                'razorpay_order_id' => $input['razorpay_order_id'],
                'razorpay_payment_id' => $input['razorpay_payment_id'],
                'razorpay_signature' => $input['razorpay_signature'],
            ];

            $this->razorpay->utility->verifyPaymentSignature($attributes);

            // Payment successful
            $startDate = Carbon::now();
            $endDate = $subscription->plan_type === 'yearly' 
                ? $startDate->copy()->addYear() 
                : $startDate->copy()->addMonth();

            $subscription->update([
                'razorpay_payment_id' => $input['razorpay_payment_id'],
                'razorpay_signature' => $input['razorpay_signature'],
                'status' => 'active',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'paid_at' => Carbon::now(),
            ]);

            // Cancel any other active subscriptions
            Subscription::where('partner_id', $subscription->partner_id)
                ->where('id', '!=', $subscription->id)
                ->where('status', 'active')
                ->update(['status' => 'cancelled']);

            return redirect()->route('partner.subscription.success', $subscription->id)
                ->with('success', 'Subscription activated successfully!');
        } catch (\Exception $e) {
            $subscription->update(['status' => 'cancelled']);
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Payment verification failed: ' . $e->getMessage());
        }
    }

    public function success($id)
    {
        $subscription = Subscription::with('partner')->findOrFail($id);
        
        if ($subscription->status !== 'active') {
            return redirect()->route('partner.subscription.index')
                ->with('error', 'Invalid subscription');
        }

        return view('partner.subscription.success', compact('subscription'));
    }
}
