<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Scheme;
use Illuminate\Http\Request;

class SchemeController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $schemes = Scheme::where('partner_id', $partnerId)->latest()->paginate(15);
        return view('partner.schemes.index', compact('schemes'));
    }

    public function create()
    {
        return view('partner.schemes.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:loan,savings',
            'interest_rate' => 'required|numeric|min:0',
            'interest_type' => 'required|in:flat,percentage',
            'duration_months' => 'nullable|integer|min:1',
            'duration_type' => 'nullable|in:daily,weekly,monthly',
            'duration_value' => 'nullable|integer|min:1|required_with:duration_type',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0|gt:min_amount',
            'processing_fee' => 'nullable|numeric|min:0',
            'insurance_fee' => 'nullable|numeric|min:0',
            'other_fee' => 'nullable|numeric|min:0',
            'late_fee_type' => 'nullable|in:percentage,fixed',
            'late_fee_value' => 'nullable|numeric|min:0|required_with:late_fee_type',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['partner_id'] = auth()->user()->partner_id;
        Scheme::create($validated);

        return redirect()->route('partner.schemes.index')->with('success', 'Scheme created successfully.');
    }

    public function show(Scheme $scheme)
    {
        $this->ensurePartnerOwnership($scheme);
        return view('partner.schemes.show', compact('scheme'));
    }

    public function edit(Scheme $scheme)
    {
        $this->ensurePartnerOwnership($scheme);
        return view('partner.schemes.edit', compact('scheme'));
    }

    public function update(Request $request, Scheme $scheme)
    {
        $this->ensurePartnerOwnership($scheme);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:loan,savings',
            'interest_rate' => 'required|numeric|min:0',
            'interest_type' => 'required|in:flat,percentage',
            'duration_months' => 'nullable|integer|min:1',
            'duration_type' => 'nullable|in:daily,weekly,monthly',
            'duration_value' => 'nullable|integer|min:1|required_with:duration_type',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0|gt:min_amount',
            'processing_fee' => 'nullable|numeric|min:0',
            'insurance_fee' => 'nullable|numeric|min:0',
            'other_fee' => 'nullable|numeric|min:0',
            'late_fee_type' => 'nullable|in:percentage,fixed',
            'late_fee_value' => 'nullable|numeric|min:0|required_with:late_fee_type',
            'status' => 'required|in:active,inactive',
        ]);

        $scheme->update($validated);

        return redirect()->route('partner.schemes.index')->with('success', 'Scheme updated successfully.');
    }

    public function destroy(Scheme $scheme)
    {
        $this->ensurePartnerOwnership($scheme);
        if ($scheme->loans()->count() > 0 || $scheme->savings()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete scheme with associated loans or savings.');
        }
        $scheme->delete();
        return redirect()->route('partner.schemes.index')->with('success', 'Scheme deleted successfully.');
    }

    private function ensurePartnerOwnership(Scheme $scheme)
    {
        if ($scheme->partner_id !== auth()->user()->partner_id) {
            abort(403);
        }
    }
}
