<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Loan;
use App\Models\Saving;
use App\Models\Repayment;
use App\Models\Customer;
use App\Models\Scheme;
use App\Models\LoanScheme;
use Carbon\Carbon;
use Illuminate\Http\Request;

class ReportsController extends Controller
{
    public function index()
    {
        return view('partner.reports.index');
    }

    public function loans(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Loan::where('partner_id', $partnerId)->with(['customer', 'loanScheme', 'scheme']);

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('disbursement_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('disbursement_date', '<=', $request->end_date);
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Scheme filter
        if ($request->filled('scheme_id')) {
            $query->where(function($q) use ($request) {
                $q->where('scheme_id', $request->scheme_id)
                  ->orWhere('loan_scheme_id', $request->scheme_id);
            });
        }

        $loans = $query->orderBy('disbursement_date', 'desc')->get();

        // Get schemes for filter dropdown
        $schemes = LoanScheme::where('partner_id', $partnerId)->where('status', 'active')->get();

        // Summary
        $summary = [
            'total_loans' => $loans->count(),
            'total_disbursed' => $loans->sum('principal_amount'),
            'total_collected' => $loans->sum('paid_amount'),
            'total_outstanding' => $loans->where('status', 'active')->sum('remaining_amount'),
            'active_loans' => $loans->where('status', 'active')->count(),
            'completed_loans' => $loans->where('status', 'completed')->count(),
        ];

        return view('partner.reports.loans', compact('loans', 'summary', 'schemes'));
    }

    public function repayments(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Repayment::where('partner_id', $partnerId)->with(['loan.customer', 'customer']);

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('payment_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('payment_date', '<=', $request->end_date);
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $repayments = $query->orderBy('payment_date', 'desc')->get();

        // Summary
        $summary = [
            'total_repayments' => $repayments->count(),
            'total_amount' => $repayments->sum('amount'),
            'total_principal' => $repayments->sum('principal_amount'),
            'total_interest' => $repayments->sum('interest_amount'),
            'total_penalty' => $repayments->sum('penalty_amount'),
        ];

        return view('partner.reports.repayments', compact('repayments', 'summary'));
    }

    public function overdue(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Loan::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->where('due_date', '<', now())
            ->with(['customer', 'scheme']);

        // Days overdue filter
        if ($request->filled('days')) {
            $days = (int) $request->days;
            $query->where('due_date', '<=', now()->subDays($days));
        }

        $loans = $query->orderBy('due_date', 'asc')->get();

        // Calculate overdue details
        $loans = $loans->map(function ($loan) {
            $loan->days_overdue = now()->diffInDays($loan->due_date);
            $loan->overdue_amount = $loan->remaining_amount;
            return $loan;
        });

        // Summary
        $summary = [
            'total_overdue' => $loans->count(),
            'total_amount' => $loans->sum('remaining_amount'),
            'avg_days_overdue' => $loans->avg('days_overdue') ? round($loans->avg('days_overdue'), 1) : 0,
        ];

        return view('partner.reports.overdue', compact('loans', 'summary'));
    }

    public function due(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Loan::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->where('due_date', '>=', now())
            ->with(['customer', 'scheme']);

        // Days ahead filter
        if ($request->filled('days')) {
            $days = (int) $request->days;
            $query->where('due_date', '<=', now()->addDays($days));
        }

        $loans = $query->orderBy('due_date', 'asc')->get();

        // Calculate due details
        $loans = $loans->map(function ($loan) {
            $loan->days_until_due = now()->diffInDays($loan->due_date, false);
            return $loan;
        });

        // Summary
        $summary = [
            'total_due' => $loans->count(),
            'total_amount' => $loans->sum('remaining_amount'),
        ];

        return view('partner.reports.due', compact('loans', 'summary'));
    }

    public function savings(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Saving::where('partner_id', $partnerId)->with(['customer', 'scheme']);

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('deposit_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('deposit_date', '<=', $request->end_date);
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Type filter
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        $savings = $query->orderBy('deposit_date', 'desc')->get();

        // Summary
        $summary = [
            'total_accounts' => $savings->where('type', 'deposit')->count(),
            'total_deposits' => $savings->where('type', 'deposit')->sum('amount'),
            'total_withdrawals' => $savings->where('type', 'withdrawal')->sum('amount'),
            'total_balance' => $savings->where('type', 'deposit')->sum('amount') - $savings->where('type', 'withdrawal')->sum('amount'),
            'active_accounts' => $savings->where('status', 'active')->count(),
        ];

        return view('partner.reports.savings', compact('savings', 'summary'));
    }

    public function collection(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Repayment::where('partner_id', $partnerId)->with(['loan.customer', 'customer']);

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('payment_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('payment_date', '<=', $request->end_date);
        }

        $repayments = $query->orderBy('payment_date', 'desc')->get();

        // Group by date
        $dailyCollections = $repayments->groupBy(function ($repayment) {
            return $repayment->payment_date->format('Y-m-d');
        })->map(function ($group) {
            return [
                'date' => $group->first()->payment_date,
                'count' => $group->count(),
                'total' => $group->sum('amount'),
                'principal' => $group->sum('principal_amount'),
                'interest' => $group->sum('interest_amount'),
                'penalty' => $group->sum('penalty_amount'),
            ];
        })->sortByDesc('date');

        // Summary
        $summary = [
            'total_collected' => $repayments->sum('amount'),
            'total_principal' => $repayments->sum('principal_amount'),
            'total_interest' => $repayments->sum('interest_amount'),
            'total_penalty' => $repayments->sum('penalty_amount'),
            'total_transactions' => $repayments->count(),
            'avg_daily_collection' => $dailyCollections->count() > 0 ? round($repayments->sum('amount') / $dailyCollections->count(), 2) : 0,
        ];

        return view('partner.reports.collection', compact('repayments', 'dailyCollections', 'summary'));
    }

    public function savingsMaturity(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $query = Saving::where('partner_id', $partnerId)
            ->where('status', 'active')
            ->whereNotNull('maturity_date')
            ->with(['customer', 'scheme']);

        // Date filters
        if ($request->filled('start_date')) {
            $query->where('maturity_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('maturity_date', '<=', $request->end_date);
        }

        // Maturity status
        if ($request->filled('maturity_status')) {
            if ($request->maturity_status === 'matured') {
                $query->where('maturity_date', '<=', now());
            } elseif ($request->maturity_status === 'upcoming') {
                $query->where('maturity_date', '>', now());
            }
        }

        $savings = $query->orderBy('maturity_date', 'asc')->get();

        // Calculate maturity details
        $savings = $savings->map(function ($saving) {
            $saving->is_matured = $saving->maturity_date <= now();
            $saving->days_until_maturity = $saving->is_matured 
                ? 0 
                : now()->diffInDays($saving->maturity_date, false);
            return $saving;
        });

        // Summary
        $summary = [
            'total_accounts' => $savings->count(),
            'matured_accounts' => $savings->where('is_matured', true)->count(),
            'upcoming_accounts' => $savings->where('is_matured', false)->count(),
            'total_maturity_amount' => $savings->sum('amount'),
        ];

        return view('partner.reports.savings-maturity', compact('savings', 'summary'));
    }

    public function balanceSheet(Request $request)
    {
        $partnerId = auth()->user()->partner_id;
        $asOnDate = $request->filled('as_on_date') ? Carbon::parse($request->as_on_date) : now();

        // Assets
        $totalDeposits = Saving::where('partner_id', $partnerId)
            ->where('type', 'deposit')
            ->where('deposit_date', '<=', $asOnDate)
            ->sum('amount');
        $totalWithdrawals = Saving::where('partner_id', $partnerId)
            ->where('type', 'withdrawal')
            ->where('deposit_date', '<=', $asOnDate)
            ->sum('amount');
        
        $assets = [
            'loans_disbursed' => Loan::where('partner_id', $partnerId)
                ->where('disbursement_date', '<=', $asOnDate)
                ->sum('principal_amount'),
            'outstanding_loans' => Loan::where('partner_id', $partnerId)
                ->where('status', 'active')
                ->where('disbursement_date', '<=', $asOnDate)
                ->sum('remaining_amount'),
            'savings_balance' => $totalDeposits - $totalWithdrawals,
        ];
        $assets['total_assets'] = array_sum($assets);

        // Liabilities
        $liabilities = [
            'savings_deposits' => Saving::where('partner_id', $partnerId)
                ->where('type', 'deposit')
                ->where('deposit_date', '<=', $asOnDate)
                ->sum('amount'),
        ];
        $liabilities['total_liabilities'] = array_sum($liabilities);

        // Income
        $income = [
            'interest_earned' => Repayment::where('partner_id', $partnerId)
                ->where('payment_date', '<=', $asOnDate)
                ->sum('interest_amount'),
            'penalty_earned' => Repayment::where('partner_id', $partnerId)
                ->where('payment_date', '<=', $asOnDate)
                ->sum('penalty_amount'),
            'processing_fees' => Loan::where('partner_id', $partnerId)
                ->where('disbursement_date', '<=', $asOnDate)
                ->sum('processing_fee'),
            'insurance_fees' => Loan::where('partner_id', $partnerId)
                ->where('disbursement_date', '<=', $asOnDate)
                ->sum('insurance_fee'),
            'other_fees' => Loan::where('partner_id', $partnerId)
                ->where('disbursement_date', '<=', $asOnDate)
                ->sum('other_fee'),
        ];
        $income['total_income'] = array_sum($income);

        // Expenses (if any in future)
        $expenses = [
            'total_expenses' => 0,
        ];

        // Net Worth
        $netWorth = $assets['total_assets'] - $liabilities['total_liabilities'] + $income['total_income'] - $expenses['total_expenses'];

        return view('partner.reports.balance-sheet', compact('assets', 'liabilities', 'income', 'expenses', 'netWorth', 'asOnDate'));
    }
}
