<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\LoanScheme;
use Illuminate\Http\Request;

class LoanSchemeController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $schemes = LoanScheme::where('partner_id', $partnerId)->latest()->paginate(15);
        return view('partner.loan-schemes.index', compact('schemes'));
    }

    public function create()
    {
        return view('partner.loan-schemes.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'interest_rate' => 'required|numeric|min:0',
            'interest_type' => 'required|in:flat,percentage',
            'duration_type' => 'required|in:daily,weekly,monthly',
            'duration_value' => 'required|integer|min:1',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0|gt:min_amount',
            'processing_fee' => 'nullable|numeric|min:0',
            'insurance_fee' => 'nullable|numeric|min:0',
            'other_fee' => 'nullable|numeric|min:0',
            'late_fee_type' => 'nullable|in:percentage,fixed',
            'late_fee_value' => 'nullable|numeric|min:0|required_with:late_fee_type',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['partner_id'] = auth()->user()->partner_id;
        LoanScheme::create($validated);

        return redirect()->route('partner.loan-schemes.index')->with('success', 'Loan scheme created successfully.');
    }

    public function show(LoanScheme $loanScheme)
    {
        $this->ensurePartnerOwnership($loanScheme);
        return view('partner.loan-schemes.show', compact('loanScheme'));
    }

    public function edit(LoanScheme $loanScheme)
    {
        $this->ensurePartnerOwnership($loanScheme);
        return view('partner.loan-schemes.edit', compact('loanScheme'));
    }

    public function update(Request $request, LoanScheme $loanScheme)
    {
        $this->ensurePartnerOwnership($loanScheme);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'interest_rate' => 'required|numeric|min:0',
            'interest_type' => 'required|in:flat,percentage',
            'duration_type' => 'required|in:daily,weekly,monthly',
            'duration_value' => 'required|integer|min:1',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0|gt:min_amount',
            'processing_fee' => 'nullable|numeric|min:0',
            'insurance_fee' => 'nullable|numeric|min:0',
            'other_fee' => 'nullable|numeric|min:0',
            'late_fee_type' => 'nullable|in:percentage,fixed',
            'late_fee_value' => 'nullable|numeric|min:0|required_with:late_fee_type',
            'status' => 'required|in:active,inactive',
        ]);

        $loanScheme->update($validated);

        return redirect()->route('partner.loan-schemes.index')->with('success', 'Loan scheme updated successfully.');
    }

    public function destroy(LoanScheme $loanScheme)
    {
        $this->ensurePartnerOwnership($loanScheme);
        if ($loanScheme->loans()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete loan scheme with associated loans.');
        }
        $loanScheme->delete();
        return redirect()->route('partner.loan-schemes.index')->with('success', 'Loan scheme deleted successfully.');
    }

    private function ensurePartnerOwnership(LoanScheme $loanScheme)
    {
        if ($loanScheme->partner_id !== auth()->user()->partner_id) {
            abort(403);
        }
    }
}
