<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Loan;
use App\Models\Customer;
use App\Models\Scheme;
use App\Models\LoanScheme;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;

class LoanController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $loans = Loan::where('partner_id', $partnerId)
            ->with(['customer', 'loanScheme', 'scheme'])
            ->latest()
            ->paginate(15);
        return view('partner.loans.index', compact('loans'));
    }

    public function create()
    {
        $partnerId = auth()->user()->partner_id;
        $customers = Customer::where('partner_id', $partnerId)->where('status', 'active')->get();
        $schemes = LoanScheme::where('partner_id', $partnerId)->where('status', 'active')->get();
        return view('partner.loans.create', compact('customers', 'schemes'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'scheme_id' => 'required|exists:loan_schemes,id',
            'principal_amount' => 'required|numeric|min:1',
            'disbursement_date' => 'required|date',
            'duration_type' => 'nullable|in:daily,weekly,monthly',
            'duration_value' => 'nullable|integer|min:1',
            'notes' => 'nullable|string',
        ]);

        $scheme = LoanScheme::findOrFail($validated['scheme_id']);
        $partnerId = auth()->user()->partner_id;

        $principal = $validated['principal_amount'];
        
        // Use scheme duration if not provided, otherwise use provided values
        $durationType = $validated['duration_type'] ?? $scheme->duration_type ?? 'monthly';
        $durationValue = (int) ($validated['duration_value'] ?? $scheme->duration_value ?? 12);
        
        // Convert duration to months for interest calculation
        $durationInMonths = match($durationType) {
            'daily' => $durationValue / 30,
            'weekly' => $durationValue / 4.33,
            'monthly' => $durationValue,
        };
        
        // Calculate interest based on scheme's interest type
        $interestRate = $scheme->interest_rate;
        $totalInterest = match($scheme->interest_type ?? 'percentage') {
            'flat' => $interestRate, // Flat amount
            'percentage' => ($principal * $interestRate * $durationInMonths) / 100, // Percentage
        };
        
        // Get fees from scheme
        $processingFee = $scheme->processing_fee ?? 0;
        $insuranceFee = $scheme->insurance_fee ?? 0;
        $otherFee = $scheme->other_fee ?? 0;
        $totalFees = $processingFee + $insuranceFee + $otherFee;
        
        // Calculate total amount: principal + interest + fees
        $totalAmount = $principal + $totalInterest + $totalFees;
        
        // Calculate installment based on duration type
        $installmentAmount = $totalAmount / $durationValue;

        $disbursementDate = Carbon::parse($validated['disbursement_date']);
        
        // Calculate maturity date based on duration type
        $maturityDate = match($durationType) {
            'daily' => $disbursementDate->copy()->addDays($durationValue),
            'weekly' => $disbursementDate->copy()->addWeeks($durationValue),
            'monthly' => $disbursementDate->copy()->addMonths($durationValue),
        };
        
        // Calculate next due date based on duration type
        $dueDate = match($durationType) {
            'daily' => $disbursementDate->copy()->addDay(),
            'weekly' => $disbursementDate->copy()->addWeek(),
            'monthly' => $disbursementDate->copy()->addMonth(),
        };

        $partner = auth()->user()->partner;
        
        $loan = Loan::create([
            'partner_id' => $partnerId,
            'customer_id' => $validated['customer_id'],
            'scheme_id' => null,
            'loan_scheme_id' => $validated['scheme_id'],
            'loan_number' => $partner->generateLoanNumber(),
            'principal_amount' => $principal,
            'interest_rate' => $interestRate,
            'processing_fee' => $processingFee,
            'insurance_fee' => $insuranceFee,
            'other_fee' => $otherFee,
            'interest_amount' => round($totalInterest, 2),
            'duration_type' => $durationType,
            'duration_value' => $durationValue,
            'monthly_installment' => round($installmentAmount, 2),
            'total_amount' => round($totalAmount, 2),
            'paid_amount' => 0,
            'remaining_amount' => round($totalAmount, 2),
            'disbursement_date' => $disbursementDate,
            'due_date' => $dueDate,
            'maturity_date' => $maturityDate,
            'status' => 'active',
            'notes' => $validated['notes'] ?? null,
        ]);

        return redirect()->route('partner.loans.show', $loan)->with('success', 'Loan created successfully.');
    }

    public function show(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        
        // Automatically apply late fees if overdue
        if ($loan->status === 'active' && $loan->due_date < now()) {
            $lateFeeService = new \App\Services\LateFeeService();
            $lateFeeService->applyLateFeesForLoan($loan);
            $loan->refresh();
        }
        
        $loan->load(['customer', 'loanScheme', 'scheme', 'repayments', 'penalties']);
        return view('partner.loans.show', compact('loan'));
    }

    public function statement(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        $loan->load(['customer', 'loanScheme', 'scheme', 'partner', 'repayments' => function($query) {
            $query->orderBy('payment_date', 'asc');
        }, 'penalties']);
        
        // Calculate totals
        $totalPaid = $loan->repayments->sum('amount');
        $totalPrincipal = $loan->repayments->sum('principal_amount');
        $totalInterest = $loan->repayments->sum('interest_amount');
        $totalPenalty = $loan->repayments->sum('penalty_amount');
        
        return view('partner.loans.statement', compact('loan', 'totalPaid', 'totalPrincipal', 'totalInterest', 'totalPenalty'));
    }

    public function schedule(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        $loan->load(['customer', 'scheme', 'partner', 'repayments' => function($query) {
            $query->orderBy('payment_date', 'asc');
        }]);
        
        // Generate repayment schedule
        $schedule = [];
        $installmentAmount = $loan->monthly_installment;
        $durationValue = $loan->duration_value;
        $durationType = $loan->duration_type;
        $disbursementDate = $loan->disbursement_date;
        $totalLoanAmount = $loan->total_amount;
        
        // Get all repayments and track which ones are allocated
        $repayments = $loan->repayments;
        $allocatedRepayments = [];
        $remainingBalance = $totalLoanAmount;
        $totalPaidSoFar = 0;
        
        for ($i = 1; $i <= $durationValue; $i++) {
            // Calculate due date based on duration type
            $dueDate = match($durationType) {
                'daily' => $disbursementDate->copy()->addDays($i),
                'weekly' => $disbursementDate->copy()->addWeeks($i),
                'monthly' => $disbursementDate->copy()->addMonths($i),
            };
            
            // Calculate installment amount (last one uses remaining balance)
            $installmentDue = ($i === $durationValue) 
                ? round($remainingBalance, 2)
                : round(min($installmentAmount, $remainingBalance), 2);
            
            // Find repayment for this installment
            $paidDate = null;
            $paidAmount = 0;
            $status = 'pending';
            
            // Calculate period for this installment
            $periodStart = ($i === 1) 
                ? $disbursementDate->copy()->subDay()
                : match($durationType) {
                    'daily' => $disbursementDate->copy()->addDays($i - 1),
                    'weekly' => $disbursementDate->copy()->addWeeks($i - 1),
                    'monthly' => $disbursementDate->copy()->addMonths($i - 1),
                };
            
            $periodEnd = $dueDate->copy()->addDays(30); // Grace period for matching
            
            // Find unallocated repayment that matches this period
            $matchingRepayment = $repayments->first(function($rep) use ($periodStart, $periodEnd, $allocatedRepayments) {
                return !in_array($rep->id, $allocatedRepayments) && 
                       $rep->payment_date->between($periodStart, $periodEnd);
            });
            
            if ($matchingRepayment) {
                $paidDate = $matchingRepayment->payment_date;
                $paidAmount = $matchingRepayment->amount;
                $allocatedRepayments[] = $matchingRepayment->id;
                $totalPaidSoFar += $paidAmount;
                
                if ($paidDate->lte($dueDate)) {
                    $status = 'paid';
                } else {
                    $status = 'overdue';
                }
            } else {
                // Check if overdue (due date passed and not paid)
                if ($dueDate->isPast()) {
                    $expectedPaid = $installmentAmount * ($i - 1);
                    if ($totalPaidSoFar < $expectedPaid + $installmentDue) {
                        $status = 'overdue';
                    }
                }
            }
            
            // Update remaining balance
            if ($paidAmount > 0) {
                $remainingBalance = max(0, $remainingBalance - $paidAmount);
            } else {
                $remainingBalance = max(0, $remainingBalance - $installmentDue);
            }
            
            $schedule[] = [
                'installment_number' => $i,
                'due_date' => $dueDate,
                'due_amount' => $installmentDue,
                'paid_date' => $paidDate,
                'paid_amount' => round($paidAmount, 2),
                'status' => $status,
                'remaining_balance' => round($remainingBalance, 2),
            ];
        }
        
        return view('partner.loans.schedule', compact('loan', 'schedule'));
    }

    public function applyLateFee(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        
        $lateFeeService = new \App\Services\LateFeeService();
        $lateFeeService->applyLateFeesForLoan($loan);
        
        return redirect()->route('partner.loans.show', $loan)
            ->with('success', 'Late fees checked and applied if applicable.');
    }

    public function edit(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        $partnerId = auth()->user()->partner_id;
        $customers = Customer::where('partner_id', $partnerId)->where('status', 'active')->get();
        $schemes = LoanScheme::where('partner_id', $partnerId)->where('status', 'active')->get();
        return view('partner.loans.edit', compact('loan', 'customers', 'schemes'));
    }

    public function update(Request $request, Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        
        $validated = $request->validate([
            'status' => 'required|in:pending,active,completed,defaulted,closed',
            'notes' => 'nullable|string',
        ]);

        $loan->update($validated);

        return redirect()->route('partner.loans.show', $loan)->with('success', 'Loan updated successfully.');
    }

    public function destroy(Loan $loan)
    {
        $this->ensurePartnerOwnership($loan);
        if ($loan->status === 'active' && $loan->remaining_amount > 0) {
            return redirect()->back()->with('error', 'Cannot delete active loan with outstanding balance.');
        }
        $loan->delete();
        return redirect()->route('partner.loans.index')->with('success', 'Loan deleted successfully.');
    }

    private function ensurePartnerOwnership(Loan $loan)
    {
        if ($loan->partner_id !== auth()->user()->partner_id) {
            abort(403);
        }
    }
}
