<?php

namespace App\Http\Controllers\Partner;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CustomerController extends Controller
{
    public function index()
    {
        $partnerId = auth()->user()->partner_id;
        $customers = Customer::where('partner_id', $partnerId)->latest()->paginate(15);
        return view('partner.customers.index', compact('customers'));
    }

    public function create()
    {
        return view('partner.customers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'father_husband_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|string|max:20',
            'address' => 'nullable|string',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'id_proof_type' => 'nullable|string|max:255',
            'id_proof_number' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $partner = auth()->user()->partner;
        $validated['partner_id'] = $partner->id;
        $validated['customer_code'] = $partner->generateCustomerCode();

        // Handle photo upload
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('customers/photos', 'public');
            $validated['photo'] = $photoPath;
        }

        Customer::create($validated);

        return redirect()->route('partner.customers.index')->with('success', 'Customer created successfully.');
    }

    public function show(Customer $customer)
    {
        $this->ensurePartnerOwnership($customer);
        $customer->load(['loans', 'savings', 'repayments']);
        return view('partner.customers.show', compact('customer'));
    }

    public function edit(Customer $customer)
    {
        $this->ensurePartnerOwnership($customer);
        return view('partner.customers.edit', compact('customer'));
    }

    public function update(Request $request, Customer $customer)
    {
        $this->ensurePartnerOwnership($customer);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|string|max:20',
            'address' => 'nullable|string',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'id_proof_type' => 'nullable|string|max:255',
            'id_proof_number' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        $customer->update($validated);

        return redirect()->route('partner.customers.index')->with('success', 'Customer updated successfully.');
    }

    public function destroy(Customer $customer)
    {
        $this->ensurePartnerOwnership($customer);
        $customer->delete();
        return redirect()->route('partner.customers.index')->with('success', 'Customer deleted successfully.');
    }

    private function ensurePartnerOwnership(Customer $customer)
    {
        if ($customer->partner_id !== auth()->user()->partner_id) {
            abort(403);
        }
    }
}
